<?PHP

#
#   FILE:  SPT--ResourceFactory.php
#
#   METHODS PROVIDED:
#       ResourceFactory()
#           - constructor
#       DuplicateResource($ResourceId)
#           - create duplicate resource and return to caller
#       (SEE ALSO:  SPT--ItemFactory.php)
#
#   AUTHOR:  Edward Almasy  
#
#   Part of the Scout Portal Toolkit
#   Copyright 2003 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--ItemFactory.php");
require_once(dirname(__FILE__)."/SPT--Resource.php");
require_once(dirname(__FILE__)."/SPT--MetadataSchema.php");


class ResourceFactory extends ItemFactory {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function __construct()
    {
        # set up item factory base class
        #$this->ItemFactory("Resource", "Resources", "ResourceId");
		parent::__construct("Resource", "Resources", "ResourceId");
    }

    # create duplicate resource and return to caller
    function DuplicateResource($ResourceId)
    {
        # create new target resource
        $DstResource = new Resource();
        
        # load up resource to duplicate
        $SrcResource = new Resource($ResourceId);
        
        # if resource to duplicate was found
        if ($SrcResource->Status() > 0)
        {
            # for each metadata field
            $Schema = new MetadataSchema();
            $Fields = $Schema->GetFields();
            foreach ($Fields as $Field)
            {
                # copy values from old resource to new resource (except for cumulative rating)
                if ($Field->Name() != "Cumulative Rating")
                {
                    $DstResource->SetByField($Field, $SrcResource->GetByField($Field, TRUE));
                }
            }
        }
        
        # return new resource to caller
        return $DstResource;
    }

    # clear or change specific qualifier for all resources
    function ClearQualifier($ObjectOrId, $NewObjectOrId = NULL)
    {
        # get ID of qualifier to clear
        if (is_object($ObjectOrId))
        {
            $QualifierId = $ObjectOrId->Id();
        }
        else
        {
            $QualifierId = $ObjectOrId;
        }

        # if new qualifier passed in
        if ($NewObjectOrId !== NULL)
        {
            # get ID of qualifier to change to
            if (is_object($NewObjectOrId))
            {
                $NewQualifierIdVal = "'".$NewObjectOrId->Id()."'";
            }
            else
            {
                $NewQualifierIdVal = "'".$NewObjectOrId."'";
            }
        }
        else
        {
            # qualifier should be cleared
            $NewQualifierIdVal = "NULL";
        }

        # for each metadata field
        $Schema = new MetadataSchema();
        $Fields = $Schema->GetFields();
        foreach ($Fields as $Field)
        {
            # if field uses qualifiers and uses item-level qualifiers
            $QualColName = $Field->DBFieldName()."Qualifier";
            if ($Field->UsesQualifiers() 
                && $Field->HasItemLevelQualifiers() 
                && $this->DB->FieldExists("Resources", $QualColName))
            {
                # set all occurrences to new qualifier value
                $this->DB->Query("UPDATE Resources"
                           ." SET ".$QualColName." = ".$NewQualifierIdVal.""
                           ." WHERE ".$QualColName." = '".$QualifierId."'");
            }
        }

        # clear or change qualifier association with controlled names
        # (NOTE: this should probably be done in a controlled name factory object)
        $this->DB->Query("UPDATE ControlledNames"
                   ." SET QualifierId = ".$NewQualifierIdVal
                   ." WHERE QualifierId = '".$QualifierId."'");

        # clear or change qualifier association with classifications
        # (NOTE: this should probably be done in a classification factory object)
        $this->DB->Query("UPDATE Classifications"
                   ." SET QualifierId = ".$NewQualifierIdVal
                   ." WHERE QualifierId = '".$QualifierId."'");
    }
    
    # return count of rated resources
    function GetRatedResourceCount()
    {
        $RatedResourceCount = $this->DB->Query("SELECT COUNT(DISTINCT ResourceId) AS ResourceCount "
                ."FROM ResourceRatings", "ResourceCount");
        return $RatedResourceCount;
    }

    # return recently released resources
    function GetRecentlyReleasedResources($Count = 10, $Offset = 0, $MaxDaysToGoBack = 90)
    {
        # assume that no resources will be found
        $Resources = array();

        # calculate cutoff date for resources
        $CutoffDate = date("Y-m-d H:i:s", strtotime($MaxDaysToGoBack." days ago"));

        # query for resource IDs
        $this->DB->Query("SELECT ResourceId FROM Resources WHERE"
                ." DateOfRecordRelease > '".$CutoffDate."'"
                ." AND ReleaseFlag = 1"
                ." AND ResourceId >= 0"
                ." ORDER BY DateOfRecordRelease DESC, DateOfRecordCreation DESC"
                ." LIMIT ".$Offset.", ".$Count);
        $ResourceIds = $this->DB->FetchColumn("ResourceId");

        # for each resource ID found
        foreach ($ResourceIds as $ResourceId)
        {
            # load resource and add to list of found resources
            $Resources[$ResourceId] = new Resource($ResourceId);
        }

        # return found resources to caller
        return $Resources;
    }
    
    # return resources sorted by specified field
    function GetResourceIdsSortedBy($FieldName, $Ascending = TRUE, $Limit = NULL)
    {
        # assume no resources will be found
        $ResourceIds = array();

        # get field
        $Schema = new MetadataSchema();
        $Field = $Schema->GetFieldByName($FieldName);
        
        # if field was found
        if ($Field != NULL)
        {
            # construct query based on field type
            switch ($Field->Type())
            {
                case MDFTYPE_TEXT:
                case MDFTYPE_PARAGRAPH:
                    $Count = $this->DB->Query("SELECT COUNT(*) AS ResourceCount"
                            ." FROM Resources WHERE "
                            .$Field->DBFieldName()." IS NOT NULL"
                            ." AND LENGTH(LTRIM(RTRIM(".$Field->DBFieldName()."))) > 0",
                            "ResourceCount");
                    if ($Count > 1)
                    {
                        $Query = "SELECT ResourceId FROM Resources"
                                ." ORDER BY ".$Field->DBFieldName()
                                .($Ascending ? " ASC" : " DESC");
                    }
                    break;

                case MDFTYPE_NUMBER:
                case MDFTYPE_TIMESTAMP:
                    $Count = $this->DB->Query("SELECT COUNT(*) AS ResourceCount"
                            ." FROM Resources WHERE "
                            .$Field->DBFieldName()." IS NOT NULL",
                            "ResourceCount");
                    if ($Count > 1)
                    {
                        $Query = "SELECT ResourceId FROM Resources"
                                ." ORDER BY ".$Field->DBFieldName()
                                .($Ascending ? " ASC" : " DESC");
                    }
                    break;

                case MDFTYPE_DATE:
                    $Count = $this->DB->Query("SELECT COUNT(*) AS ResourceCount"
                            ." FROM Resources WHERE "
                            .$Field->DBFieldName()."Begin IS NOT NULL",
                            "ResourceCount");
                    if ($Count > 1)
                    {
                        $Query = "SELECT ResourceId FROM Resources"
                                ." ORDER BY ".$Field->DBFieldName()."Begin"
                                .($Ascending ? " ASC" : " DESC");
                    }
                    break;
            }
            
            # if appropriate query was found
            if (isset($Query))
            {
                # if limited number of results were requested
                if ($Limit !== NULL)
                {
                    # add limit to query
                    $Query .= " LIMIT ".intval($Limit);
                }
                
                # perform query and retrieve resource IDs
                $this->DB->Query($Query);
                $ResourceIds = $this->DB->FetchColumn("ResourceId");
            }            
        }
        
        # return resource IDs to caller
        return $ResourceIds;
    }

    # get date/time of last resource modification (returned as Unix timestamp)
    function GetTimestampOfLastResourceModification($OnlyReleasedResources = TRUE)
    {
        $LastChangeDate = $this->DB->Query(
                "SELECT MAX(DateLastModified) AS LastChangeDate"
                ." FROM Resources".($OnlyReleasedResources ? " WHERE ReleaseFlag = 1" : ""),
                "LastChangeDate");
        return ($LastChangeDate ? strtotime($LastChangeDate) : NULL);
    }

	# get list of possible field names for resources
	function GetPossibleFieldNames()
	{
		# retrieve field names from schema
		$FieldNames = array();
		$Schema = new MetadataSchema();
		$Fields = $Schema->GetFields();
		foreach ($Fields as $Field)
		{
			$FieldNames[$Field->Id()] = $Field->Name();
		}
		
		# return field names to caller
		return $FieldNames;
	}

    # find resources with values that match those specified
    # (index of $ValuesToMatch is field IDs)
    function GetMatchingResources($ValuesToMatch)
    {
        # start out assuming we won't find any resources
        $Resources = array();

        # for each value
        $Schema = new MetadataSchema();
        $Fields = $Schema->GetFields(MDFTYPE_TEXT | MDFTYPE_PARAGRAPH 
                | MDFTYPE_NUMBER | MDFTYPE_DATE | MDFTYPE_TIMESTAMP | MDFTYPE_FLAG);
        $LinkingTerm = "";
        $Condition = "";
        foreach ($ValuesToMatch as $FieldId => $Value)
        {
            # if field can be used for comparison
            if (isset($Fields[$FieldId]))
            {
                # add comparison to condition
                $Condition .= $LinkingTerm.$Fields[$FieldId]->DBFieldName()
                        ." = '".addslashes($Value)."'";
                $LinkingTerm = " AND ";
            }
        }

        # if there were valid conditions
        if (strlen($Condition))
        {
            # build query statment
            $Query = "SELECT ResourceId FROM Resources WHERE ".$Condition;

            # execute query to retrieve matching resource IDs
            $this->DB->Query($Query);
            $ResourceIds = $DB->FetchColumn("ResourceId");

            # retrieve resource objects
            foreach ($ResourceIds as $Id)
            {
                $Resources[$Id] = new Resource($Id);
            }
        }

        # return any resources found to caller
        return $Resources;
    }


    # ---- PRIVATE INTERFACE -------------------------------------------------

}


?>
