<?PHP
#
#   FILE:  SPT--EditResourceComplete.php
#
#   FUNCTIONS PROVIDED:
#       FunctionName($Parameter, $Parameter)
#           - brief description of function supplied for use in HTML file
#
#   FUNCTIONS EXPECTED:
#       FunctionName($Parameter, $Parameter)
#           - brief description of function that HTML file must supply
#
#   OPTIONAL FUNCTIONS:
#       FunctionName($Parameter, $Parameter)
#           - brief description of function that HTML file may supply
#
#   FORM FIELDS EXPECTED:
#       F_FormFieldName
#           - brief description of form field purpose and/or content
#
#   Part of the Scout Portal Toolkit
#   Copyright 2007 Internet Scout
#   http://scout.wisc.edu
#

require_once("include/SPT--Common.php");
require_once("include/SPT--MetadataSchema.php");
require_once("include/SPT--Resource.php");
require_once("include/SPT--SearchEngine.php");
require_once("include/SPT--Recommender.php");
PageTitle("Resource Editing Complete");

CheckAuthorization(PRIV_RESOURCEADMIN, PRIV_MYRESOURCEADMIN, PRIV_RELEASEADMIN);


# ----- CONFIGURATION  -------------------------------------------------------

# ----- EXPORTED FUNCTIONS ---------------------------------------------------
# (functions intended for use in corresponding HTML file)

# ----- LOCAL FUNCTIONS ------------------------------------------------------
# (functions intended for use only within this file)

# error codes  (NOTE:  Must match codes in SPT--EditResourceCommon.html!!!)
define("ERR_FILEUPLOADERROR", 1);
define("ERR_ZEROLENGTH", 2);
define("ERR_IMAGEUPLOADERROR", 3);
define("ERR_UNSUPPORTEDIMAGEFORMAT", 4);
define("ERR_UNKNOWNIMAGETYPE", 5);
define("ERR_REQUIREDFIELDEMPTY", 6);

# return back to editing page with error info
function ReturnWithErrors($Resource, $Fields, $Error, $ErrOne = NULL, $ErrTwo = NULL)
{
    if (is_array($Fields))
    {
        $FieldName = "Required Fields";
        $FieldInfo = "";
        foreach ($Fields as $Field)
        {
            $FieldInfo .= (strlen($FieldInfo) ? "-" : "").$Field->Id();
        }
    }
    else
    {
        $FieldName = $Fields->Name();
        $FieldInfo = $Fields->Id();
    }
    PrintAutoRefreshPage("Error Encountered with ".$FieldName,
            "SPT--EditResource.php?ID=".$Resource->Id()
                    ."&ER=".$Error."&EF=".$FieldInfo
                    .($ErrOne ? "&E1=".urlencode($ErrOne) : "")
                    .($ErrTwo ? "&E2=".urlencode($ErrTwo) : ""));
    exit(0);
}

# update search and recommender system DBs (if dynamic updates enabled)
function UpdateSearchAndRecommender($Resource)
{
    global $G_SysConfig;
    if ($G_SysConfig->SearchDBEnabled())
    {
        $SearchEngine = new SPTSearchEngine();
        $SearchEngine->UpdateForResource($Resource->Id());
    }
    if ($G_SysConfig->RecommenderDBEnabled())
    {
        $Recommender = new SPTRecommender();
        $Recommender->UpdateForItem($Resource->Id());
    }
}

# save values from editing form
# (returns array of any unfilled required fields)
function UpdateResource($Resource, $User)
{
    # for each metadata field
    $Schema = new MetadataSchema();
    $Fields = $Schema->GetFields(NULL, MDFORDER_EDITING);
    $EmptyFields = array();
    foreach ($Fields as $Field)
    {
        # if user has permission to edit field
        if ($Resource->UserCanEditField($User, $Field))
        {
            switch ($Field->Type())
            {
                case MDFTYPE_TEXT:
                case MDFTYPE_NUMBER:
                case MDFTYPE_DATE:
                case MDFTYPE_TIMESTAMP:
                case MDFTYPE_PARAGRAPH:
                case MDFTYPE_FLAG:
                    # if we have a value from the form for this field
                    # (check necessary because user may push Save button
                    #       before form has finished loading)
                    if (isset($_POST["F_".$Field->DBFieldName()]))
                    {
                        # update release date if release flag was toggled to true
                        $NewValue = trim($_POST["F_".$Field->DBFieldName()]);
                        $NewValue = strip_tags($NewValue, 
                                "<b><i><a><br><sub><sup><strike>");
                        if (($Field->Name() == "Release Flag")
                            && (!$Resource->Get("Release Flag"))
                            && ($NewValue == "1"))
                        {
                            $Resource->Set("Date Of Record Release", date("Y-m-d H:i:s"));
                        }

                        # filter date field values for validity
                        if ($Field->Type() & (MDFTYPE_DATE|MDFTYPE_TIMESTAMP))
                        {
                            $TestDate = new Date($NewValue);
                            if ($TestDate->Precision() == 0) {  $NewValue = "";  }
                        }

                        # if value was supplied or field is not required
                        if (strlen($NewValue) || $Field->Optional()) 
                        {
                            # save field
                            $Resource->Set($Field, $NewValue);
                        }
                        else
                        {
                            # add field to error list
                            $EmptyFields[] = $Field;
                        }
                    }
                    break;

                case MDFTYPE_TREE:
                case MDFTYPE_CONTROLLEDNAME:
                    # while there are form values for this field
                    $ValueCount = $Field->GetCountOfPossibleValues();
                    $InterfaceToggleThreshold = 250;
                    $Factory = $Field->GetFactory();
                    $BaseFieldName = "D_".$Field->DBFieldName()."_";
                    $FieldIndex = 1;
                    $ValuesToSet = array();
                    while (isset($_POST[$BaseFieldName.$FieldIndex]))
                    {
                        # retrieve value from form field
                        $Value = $_POST[$BaseFieldName.$FieldIndex];

                        # if search was used (returns names instead of indexes)
                        if ($ValueCount >= $InterfaceToggleThreshold)
                        {
                            # convert value to index
                            if (strlen($Value))
                            {
                                $Item = $Factory->GetItemByName($Value);
                                if ($Item)
                                {
                                    $Value = $Item->Id();
                                }
                                else
                                {
                                    $Value = -1;
                                }
                            }
                            else
                            {
                                $Value = -1;
                            }
                        }

                        # if form value appears valid
                        if ($Value >= 0)
                        {
                            # add value to list of those to be set
                            # (set method expects IDs to appear as indexes)
                            $ValuesToSet[$Value] = 1;
                        }

                        # move to next form value
                        $FieldIndex++;
                    }

                    # if value found to set or field is not required
                    if (count($ValuesToSet) || $Field->Optional()) 
                    {
                        # clear any existing values for this field
                        $Resource->ClearByField($Field);

                        # if values found to set
                        if (count($ValuesToSet))
                        {
                            # set values in resource
                            $Resource->Set($Field, $ValuesToSet);
                        }
                    }
                    else
                    {
                        # add field to error list
                        $EmptyFields[] = $Field;
                    }
                    break;

                case MDFTYPE_OPTION:
                    # clear any existing values for this field
                    $Resource->ClearByField($Field);

                    # if field allows multiple values
                    $ValuesToSet = array();
                    if ($Field->AllowMultiple())
                    {
                        # retrieve possible values for this field
                        $PossibleValues = $Field->GetPossibleValues();

                        # for each possible value
                        foreach ($PossibleValues as $ValueId => $ValueName)
                        {
                            # if form field is set for value
                            if (isset($_POST["D_".$Field->DBFieldName()."_".$ValueId]))
                            {
                                # add value to list of those to be set
                                $ValuesToSet[$ValueId] = 1;
                            }
                        }
                    }
                    else
                    {
                        # retrieve value for this field (if available)
                        if (isset($_POST["F_".$Field->DBFieldName()]))
                        {
                            $ValuesToSet[$_POST["F_".$Field->DBFieldName()]] = 1;
                        }
                    }

                    # if value found to set or field is not required
                    if (count($ValuesToSet) || $Field->Optional()) 
                    {
                        # clear any existing values for this field
                        $Resource->ClearByField($Field);

                        # if values found to set
                        if (count($ValuesToSet))
                        {
                            # set values in resource
                            $Resource->Set($Field, $ValuesToSet);
                        }
                    }
                    else
                    {
                        # add field to error list
                        $EmptyFields[] = $Field;
                    }
                    break;

                case MDFTYPE_USER:
                    # (this type not currently editable)
                    break;

                case MDFTYPE_IMAGE:
                case MDFTYPE_FILE:
                    # (these types handled via special upload mechanisms)
                    break;

                default:
                    break;
            }

            # set flag to indicate that user was able to modify record
            $FieldsEdited = TRUE;
        }
    }

    # if user was able to modify record
    if ($FieldsEdited)
    {
        # update modification time stamp and user ID
        $Resource->Set("Date Last Modified", "now");
        $Resource->Set("Last Modified By Id", $User);

        # update search and recommender DBs if configured to do so
        UpdateSearchAndRecommender($Resource);
    }

    # return list of any empty required fields to caller
    return $EmptyFields;
}

# process File upload requests
function UploadFiles($Resource, $User)
{
    # for each metadata field that might have an uploaded file
    $Schema = new MetadataSchema();
    $Fields = $Schema->GetFields(MDFTYPE_FILE);
    foreach ($Fields as $Field)
    {
        # if field is modifiable by specified user
        #       and we have an uploaded file for this field
        if ($Resource->UserCanEditField($User, $Field)
                && isset($_FILES[$Field->DBFieldName()]["tmp_name"])
                && is_uploaded_file($_FILES[$Field->DBFieldName()]["tmp_name"]))
        {
            # save uploaded file
            $TmpFileName = $_FILES[$Field->DBFieldName()]["tmp_name"];
            $NewFile = new File($TmpFileName, $Resource->Id(), $Field->Id(),
                    $_FILES[$Field->DBFieldName()]["name"]);

            # if file save failed
            if ($NewFile->Status() != FILESTAT_OK)
            {
                # set error message and error out
                switch ($NewFile->Status())
                {
                    case FILESTAT_ZEROLENGTH:
                        $Error = ERR_ZEROLENGTH;
                        break;

                    default:
                        $Error = ERR_FILEUPLOADERROR;
                        break;
                }
                $ErrParamOne = $_FILES[$Field->DBFieldName()]['name'];
                $ErrParamTwo = $NewFile->Status();
                ReturnWithErrors($Resource, $Field, $Error, $ErrParamOne, $ErrParamTwo);
            }

            # remove temp file
            unlink($TmpFileName);
        }
    }
}

# process image upload requests
function UploadImages($Resource, $User)
{
    # check to make sure image storage directories are okay
    if ($ErrorMsgs = SPTImage::CheckDirectories())
    {
        # NOTE:  NEED TO HANDLE IMAGE DIR ERRORS HERE!!!
        exit(1);
    }

    # for each metadata field that might have an uploaded image
    $Schema = new MetadataSchema();
    $Fields = $Schema->GetFields(MDFTYPE_IMAGE);
    foreach ($Fields as $Field)
    {
        # if field is modifiable by specified user
        #       and we have an uploaded file for this field
        if ($Resource->UserCanEditField($User, $Field)
                && isset($_FILES[$Field->DBFieldName()]["tmp_name"])
                && is_uploaded_file($_FILES[$Field->DBFieldName()]["tmp_name"]))
        {
            # create temp copy of file with correct name
            $TempFile = "TempStorage/".$_FILES[$Field->DBFieldName()]['name'];
            copy($_FILES[$Field->DBFieldName()]["tmp_name"], $TempFile);

            # create new Image object from uploaded file
            $Image = new SPTImage($TempFile,
                    $Field->MaxPreviewWidth(), $Field->MaxPreviewHeight(),
                    $Field->MaxThumbnailWidth(), $Field->MaxThumbnailHeight());

            # if file save failed
            if ($Image->Status() != AI_OKAY)
            {
                # set error message and error out
                switch ($Image->Status())
                {
                    case AI_UNKNOWNTYPE:
                    case AI_UNSUPPORTEDFORMAT:
                        $Error = ($Image->Status() == AI_UNSUPPORTEDFORMAT)
                                ? ERR_UNSUPPORTEDIMAGEFORMAT : ERR_UNKNOWNIMAGETYPE;
                        $ErrParamOne = $_FILES[$Field->DBFieldName()]['name'];
                        ReturnWithErrors($Resource, $Field, $Error, $ErrParamOne);
                        break;

                    default:
                        $Error = ERR_IMAGEUPLOADERROR;
                        $ErrParamOne = $_FILES[$Field->DBFieldName()]['name'];
                        $ErrParamTwo = $Image->Status();
                        ReturnWithErrors($Resource, $Field, $Error, 
                                $ErrParamOne, $ErrParamTwo);
                        break;
                }
            }
            else
            {
                # attach image to resource
                $Resource->Set($Field, $Image->Id());
            }
        }
    }
}


# ----- MAIN -----------------------------------------------------------------
# PrintForDebug("_POST", $_POST);
# exit(1);

# retrieve resource being edited
if (isset($_POST["H_ResourceId"]))
{
    $G_Resource = new Resource(intval($_POST["H_ResourceId"]));
}

# if we could not find resource with supplied ID
if (!isset($G_Resource) || !is_object($G_Resource) || ($G_Resource->Status() != 1))
{
    # COULD NOT FIND RESOURCE WITH SPECIFIED ID
}
else
{
    # take action based on which submit button was pressed 
    switch (isset($_POST["Submit"]) ? $_POST["Submit"] : NULL)
    {
        case "Add":
            # save changes to record
            $EmptyFields = UpdateResource($G_Resource, $G_User);


            # if some required fields were not filled out
            if (count($EmptyFields))
            {
                # return to editing page with error
                ReturnWithErrors($G_Resource, $EmptyFields, ERR_REQUIREDFIELDEMPTY);
            }
            else
            {
                # convert temporary record to permanent
                $G_Resource->IsTempResource(FALSE);

                # update search and recommender DBs (if configured to do so)
                UpdateSearchAndRecommender($G_Resource);
            }
            break;

        case "Save":
            # save changes to record
            $EmptyFields = UpdateResource($G_Resource, $G_User);

            # if some required fields were not filled out
            if (count($EmptyFields))
            {
                # return to editing page with error
                ReturnWithErrors($G_Resource, $EmptyFields, ERR_REQUIREDFIELDEMPTY);
            }
            break;

        case "Duplicate":
            # create duplicate resource
            $Factory = new ResourceFactory();
            $DupResource = $Factory->DuplicateResource($G_Resource->Id());

            # modify likely field to indicate resource is duplicate
            $Schema = new MetadataSchema();
            if ($Schema->GetFieldByName("Title"))
            {
                $DupResource->Set("Title", 
                        $DupResource->Get("Title")." [DUPLICATE]");
            }
            elseif ($Schema->GetFieldByName("Description"))
            {
                $DupResource->Set("Description", 
                        $DupResource->Get("Description")." [DUPLICATE]");
            }

            # return to editing page with duplicate
            PrintAutoRefreshPage("Resource Duplicated",
                    "SPT--EditResource.php?ID=".$DupResource->Id());
            break;

        case "Delete":
            # if delete confirmation already requested
            if (isset($_POST["H_DeletionRequested"]))
            {
                # set flag to indicate resource should be deleted
                $G_ResourceDeleted = TRUE;
            }
            else
            {
                # save changes to record
                UpdateResource($G_Resource, $G_User);

                # set flag to indicate need to ask for delete confirmation
                $G_DeleteRequested = TRUE;
            }
            break;

        case "Cancel":
            # if cancelling resource deletion
            if (isset($_POST["H_DeletionRequested"]))
            {
                # set flag to indicate that deletion was cancelled
                $G_DeletionCancelled = TRUE;
            }
            else
            {
                # if this was a new resource being added
                if ($G_Resource->IsTempResource())
                {
                    # set flag to indicate that adding a new resource was cancelled
                    $G_AddingCancelled = TRUE;
                }
                else
                {
                    # set flag to indicate that editing was cancelled
                    $G_EditingCancelled = TRUE;
                }
            }
            break;

        case "Upload File":
            # save changes to record
            UpdateResource($G_Resource, $G_User);

            # process file upload requests
            UploadFiles($G_Resource, $G_User);

            # return to editing page
            PrintAutoRefreshPage("File Uploaded",
                    "SPT--EditResource.php?ID=".$G_Resource->Id());
            exit(0);
            break;

        case "Upload Image":
            # save changes to record
            UpdateResource($G_Resource, $G_User);

            # process file upload requests
            UploadImages($G_Resource, $G_User);

            # return to editing page
            PrintAutoRefreshPage("Image Uploaded",
                    "SPT--EditResource.php?ID=".$G_Resource->Id());
            exit(0);
            break;

        default:
            # save changes to record
            UpdateResource($G_Resource, $G_User);

            # for each controlled name and tree field
            $Schema = new MetadataSchema();
            $Fields = $Schema->GetFields(MDFTYPE_CONTROLLEDNAME | MDFTYPE_TREE);
            foreach ($Fields as $Field)
            {
                # if user pressed search button for field
                if (isset($_POST["SearchField_".$Field->Id()]))
                {
                    # jump to search page for field
                    PrintAutoRefreshPage("Search for ".$Field->Name(),
                            "SPT--EditResourceSearchName.php?ID="
                            .$G_Resource->Id()."&MF=".$Field->Id());
                    exit(0);
                }
            }

            # for each file or image field
            $Schema = new MetadataSchema();
            $Fields = $Schema->GetFields(MDFTYPE_FILE | MDFTYPE_IMAGE);
            foreach ($Fields as $Field)
            {
                # if user has permission to modify field
                if ($G_Resource->UserCanEditField($G_User, $Field))
                {
                    # for each existing file for field
                    $Files = $G_Resource->GetByField($Field, TRUE);
                    if (is_object($Files)) {  $Files = array($Files);  }
                    foreach ($Files as $File)
                    {
                        # if user pressed delete button for file
                        if (isset($_POST["DeleteFile_".$File->Id()]))
                        {
                            # delete file
                            $File->Delete();
    
                            # return to edit page with appropriate message
                            PrintAutoRefreshPage("Deleted File in ".$Field->Name(),
                                    "SPT--EditResource.php?ID="
                                    .$G_Resource->Id());
                            exit(0);
                        }
                        # else if user pressed delete button for image
                        elseif (isset($_POST["DeleteImage_".$File->Id()]))
                        {
                            # clear image
                            $G_Resource->ClearByField($Field);
    
                            # return to edit page with appropriate message
                            PrintAutoRefreshPage("Deleted Image in ".$Field->Name(),
                                    "SPT--EditResource.php?ID="
                                    .$G_Resource->Id());
                            exit(0);
                        }
                    }
                }
            }
            break;
    }
}

# load fields for display usage
$Schema = new MetadataSchema();
$G_Fields = $Schema->GetFields(NULL, MDFORDER_EDITING);

include("include/SPT--StandardHtmlPageLoad.php");

?>
