<?PHP

#
#   FILE:  SPT--ControlledName.php
#
#   METHODS PROVIDED:
#       ControlledName($NameId, $Name = NULL, $FieldId = NULL)
#           - constructor
#       Delete()
#           - remove controlled name (and associations) from database
#       Id()
#       FullName()
#       Variant()
#       FieldId($NewValue = DB_NOVALUE)
#       LinkString($NewValue = DB_NOVALUE)
#       QualiferId($NewValue = DB_NOVALUE)
#           - get/set attributes
#
#   AUTHOR:  Barry Wiegan 
#
#   Part of the Scout Portal Toolkit
#   Copyright 2004 Internet Scout Project
#   http://scout.wisc.edu
#

require_once(dirname(__FILE__)."/SPT--SPTDatabase.php");

# error status codes
define("CNSTAT_OK",             0);
define("CNSTAT_INVALIDID",      1);
define("CNSTAT_EXISTS",         3);

class ControlledName {

    # ---- PUBLIC INTERFACE --------------------------------------------------

    # object constructor
    function ControlledName($NameId, $Name = NULL, $FieldId = NULL, 
                    $QualifierId = "NULL", $VariantName = NULL)
    {
        # assume everything will turn out okay
        $this->ErrorStatus = CNSTAT_OK;
        
        # create DB handle for our use
        $this->DB = new SPTDatabase();
        $DB =& $this->DB;

        $Name = addslashes($Name);

        # look for passed in name and type
        if (!empty($Name) && !empty($FieldId))
        {
            $DB->Query("SELECT * FROM ControlledNames".
                   " WHERE ControlledName = \"".addslashes($Name)."\"".
                   " AND FieldId = ".intval($FieldId));

            while ($this->DBFields = $DB->FetchNextRowArray())
            {
                # this controlled name already exists
                if ($this->DBFields["ControlledName"] == stripslashes($Name))
                {
                    $this->ErrorStatus = CNSTAT_EXISTS;
                    $NameId = $this->DBFields["ControlledNameId"];
                
                    # cache the variant name separately
                    $VN = $DB->Query("SELECT VariantName FROM VariantNames".
                        " WHERE ControlledNameId = ".
                        $this->DBFields["ControlledNameId"], "VariantName"); 

                    $this->DBFields["VariantName"] = $VN;
                    break;
                }
            }
            # controlled name not found, create it
            if ($this->ErrorStatus == CNSTAT_OK) 
            {
                # add new controlled name
                $DB->Query("INSERT INTO ControlledNames ".
                        "(FieldId, ControlledName, QualifierId)".
                        " VALUES (".$FieldId.", '".addslashes($Name)."', $QualifierId)");

                # get name ID  for new controlled name
                $NameId = $DB->LastInsertId("ControlledNames");

                # check for Variant
                if (!empty($VariantName))
                {
                    $DB->Query("Insert into VariantNames ".
                        "(ControlledNameId, VariantName) ".
                        "values ($NameId, '".addslashes($VariantName)."') ");
                }
            }
        }
        # Name Id passed in, look it up
        if (!empty($NameId) && $NameId != -1)
        {
            $DB->Query("SELECT * FROM ControlledNames".
                   " WHERE ControlledNameId = ".intval($NameId));
            $this->DBFields = $DB->FetchRow();

            # cache the variant name separately
            $VN = $DB->Query("SELECT VariantName FROM VariantNames".
                   " WHERE ControlledNameId = ".intval($NameId), "VariantName"); 

            $this->DBFields["VariantName"] = $VN;
        }

        # save supplied or generated controlled name ID
        $this->Id = intval($NameId);

        # set error status if controlled name info not loaded
        if ($this->DBFields["ControlledNameId"] != $this->Id)
        {
            $this->ErrorStatus = CNSTAT_INVALIDID;
        }
    }
    
    # check success of last call
    function Status()   {  return $this->ErrorStatus;  }

    # get attributes
    function Id()       {  return $this->Id;  }

    # get/set attributes
    function Name($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("ControlledName", $NewValue);  }
    function VariantName($NewValue = DB_NOVALUE)
            {  return $this->UpdateVariantValue("VariantName", $NewValue);  }
    function FieldId($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("FieldId", $NewValue);  }
    function QualifierId($NewValue = DB_NOVALUE)
            {  return $this->UpdateValue("QualifierId", $NewValue);  }
    function Variant($NewValue = DB_NOVALUE)
            {  return $this->VariantName($NewValue);  }

    # get/set associated qualifier
    function Qualifier($NewValue = DB_NOVALUE)
    {
        # if new qualifier supplied
        if ($NewValue !== DB_NOVALUE)
        {
            # set new qualifier ID
            $this->QualifierId($NewValue->Id());

            # use new qualifier for return value
            $Qualifier = $NewValue;
        }
        else
        {
            # if qualifier is available
            if ($this->QualifierId() !== NULL)
            {
                # create qualifier object using stored ID
                $Qualifier = new Qualifier($this->QualifierId());

                # if ID was zero and no name available for qualifieR
                # (needed because some controlled name records in DB
                #       have 0 instead of NULL when no controlled name assigned)
                # (NOTE:  this is problematic if there is a qualifier with an
                #       ID value of 0!!!)
                if (($this->QualifierId() == 0) && !strlen($Qualifier->Name()))
                {
                    # return NULL to indicate no qualifier
                    $Qualifier = NULL;
                }
            }
            else
            {
                # return NULL to indicate no qualifier
                $Qualifier = NULL;
            }
        }

        # return qualifier to caller
        return $Qualifier;
    }

    # see if controlled name is in use
    function InUse()
    {
        return $this->DB->Query("SELECT COUNT(*) AS Count FROM ".
                "ResourceNameInts WHERE ControlledNameId = ".$this->Id, "Count");
    }

    # change all usage of this name to new name
    function RemapTo($NewNameId)
    {
        $this->DB->Query("UPDATE ResourceNameInts SET ControlledNameId = "
                .intval($NewNameId)." WHERE ControlledNameId = ".$this->Id);
    }

    # remove controlled name (and accompanying associations) from database
    function Delete($DeleteIfHasResources = FALSE)
    {
        $DB =& $this->DB;

        if ($DeleteIfHasResources || !$this->InUse())
        {
            # delete this controlled name
            $DB->Query("DELETE FROM ControlledNames WHERE ControlledNameId=".
                $this->Id);

            # delete associated variant name
            $DB->Query("DELETE FROM VariantNames WHERE ControlledNameId=".
                $this->Id);

            if ($DeleteIfHasResources)
            {
                $DB->Query("DELETE FROM ResourceNameInts WHERE ".
                   "ControlledNameId=".$this->Id);
            }
        }
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    var $DB;
    var $DBFields;
    var $Id;
    var $ErrorStatus;

    # convenience function to supply parameters to Database->UpdateValue()
    function UpdateValue($FieldName, $NewValue)
    {
        return $this->DB->UpdateValue("ControlledNames", $FieldName, 
                $NewValue, "ControlledNameId = ".$this->Id,
                $this->DBFields, TRUE);
    }
    
    # convenience function for VariantNames table
    function UpdateVariantValue($FieldName, $NewValue)
    {
        # update require_once("include/SPT--Common.php");
        if (!empty($NewValue))
        {
            # see if variant name exists for the controlled Name
            $this->DB->Query("SELECT * from VariantNames WHERE ".
                        "ControlledNameId = ".$this->Id);

            # variant name exists so do an update
            if ($this->DB->NumRowsSelected() > 0)
            {
                return $this->DB->UpdateValue("VariantNames", 
                            $FieldName, $NewValue,
                            "ControlledNameId = ".$this->Id,
                            $this->DBFields, TRUE);
            }
            # no variant name so do an insert
            else if ($NewValue != DB_NOVALUE)
            {
                $this->DB->Query("INSERT into VariantNames ".
                        "(VariantName, ControlledNameId) VALUES ".
                        "(\"$NewValue\",".$this->Id.")");
            }
        }
        # delete variant name
        else
        {
            $this->DB->Query("Delete from VariantNames where ".
                "ControlledNameId = ".$this->Id);
        }
    }
}

?>
