<?PHP
#
#   FILE:  Rules_Rule.php  (Rules plugin)
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Class representing an individual rule in the Rules plugin.
*/
class Rules_Rule extends Item
{
    # ---- PUBLIC INTERFACE --------------------------------------------------

    /** Action types. */
    const ACTION_NONE = 0;
    const ACTION_SENDEMAIL = 1;

    /** Rule check frequencies.  (Values are in minutes.) */
    const CHECKFREQ_ONCHANGE = -1;
    const CHECKFREQ_HOURLY = 60;
    const CHECKFREQ_DAILY = 1440;
    const CHECKFREQ_WEEKLY = 10080;

    /**
    * Create new rule.
    * @param object $SearchParams Search parameters to use when checking rule.
    * @param int $Action Action to take when items match rule.
    * @param array $ActionParams Parameters for action.  (OPTIONAL)
    * @return object New rule object.
    */
    public static function Create($SearchParams, $Action, $ActionParams = array())
    {
        # instantiate new Rule object
        $InitialValues = array(
                "LastChecked" => date("Y-m-d H:i:s"),
                "CheckFrequency" => self::CHECKFREQ_HOURLY,
                "DateCreated" => date("Y-m-d H:i:s"),
                "CreatedBy" => $GLOBALS["G_User"]->Id(),
                );
        $Rule = parent::CreateWithValues($InitialValues);
        $Rule->SearchParameters($SearchParams);
        $Rule->Action($Action);
        $Rule->ActionParameters($ActionParams);

        # set base list of found items
        $MatchingIds = $Rule->Check();
        $Rule->LastMatchingIds($MatchingIds);

        # return new Rule object to caller
        return $Rule;
    }

    /**
    * Get/set whether the rule is enabled.
    * @param bool $NewValue TRUE to enable, or FALSE to disable.  (OPTIONAL)
    * @return bool TRUE if rule is enabled, otherwise FALSE.
    */
    public function Enabled($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Enabled", $NewValue);
    }

    /**
    * Get/set how often to check the rule.
    * @param int $NewValue New frequency, in minutes.  (OPTIONAL)
    * @return int Current frequency.
    */
    public function CheckFrequency($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("CheckFrequency", $NewValue);
    }

    /**
    * Get/set search parameters used to identify matching resources.
    * @param object $NewValue New parameters (SearchParameterSet).  (OPTIONAL)
    * @return object Current search parameters (SearchParameterSet).
    */
    public function SearchParameters($NewValue = NULL)
    {
        $NewStoredValue = ($NewValue === NULL) ? DB_NOVALUE
                : $NewValue->Data();
        $StoredValue = $this->UpdateValue("SearchParams", $NewStoredValue);
        return new SearchParameterSet(
                strlen($StoredValue) ? $StoredValue : NULL);
    }

    /**
    * Get/set action to be taken for items that match rule.
    * @param int $NewValue New action.  (OPTIONAL)
    * @return int Current action.
    */
    public function Action($NewValue = DB_NOVALUE)
    {
        return $this->UpdateValue("Action", $NewValue);
    }

    /**
    * Get/set parameters for action to be taken for items that match rule.
    * @param array $NewValue New parameter settings, with parameter names
    *       for index.  (OPTIONAL)
    * @return array Current parameters, with parameter names for index.
    */
    public function ActionParameters($NewValue = DB_NOVALUE)
    {
        if ($NewValue != DB_NOVALUE)
        {
            $NewValue = serialize($NewValue);
        }
        $NewValue = @unserialize($this->UpdateValue("ActionParams", $NewValue));
        return ($NewValue === FALSE) ? array() : $NewValue;
    }

    /**
    * Check rule and report what resources match.
    * @return array Array of arrays of IDs of items found for rule,
    *       indexed (at the top level) by user ID.
    */
    public function Check()
    {
        # substitute last check date/time into search parameters as needed
        $SearchParams = $this->SearchParameters();
        $SearchParams->ReplaceSearchString("/^@ LAST_CHECKED\$/",
                "@ ".date("Y-m-d H:i:s", $this->LastChecked()));

        # perform search for matching resources
        $ItemIds = $this->Search($SearchParams);

        # update "last checked" date
        $this->LastChecked("NOW");

        # return list of new matching resources to caller
        return $ItemIds;
    }

    /**
    * Perform action for rule.
    * @param array $ItemIds Array of arrays of item IDs, indexed by user ID.
    */
    public function Act($ItemIds)
    {
        switch ($this->Action())
        {
            case self::ACTION_SENDEMAIL:
                $this->PerformAction_SendEmail(
                        $this->ActionParameters(), $ItemIds);
                break;

            default:
                throw new Exception("Unknown action (".$this->Action()
                        .") for rule \"".$this->Name()."\" (".$this->Id().").");
                break;
        }
        $this->LastMatchingIds($ItemIds);
    }

    /**
    * Get/set per user list of item IDs that last matched for this rule.
    * @param array $NewValue Array of arrays of item IDs, indexed by user ID.  (OPTIONAL)
    * @return array Array of arrays of item IDs, indexed by user ID.
    */
    public function LastMatchingIds($NewValue = NULL)
    {
        $NewStoredValue = ($NewValue === NULL) ? DB_NOVALUE
                : serialize($NewValue);
        $StoredValue = $this->UpdateValue("LastMatchingIds", $NewStoredValue);
        $Value = strlen(trim($StoredValue)) ? unserialize($StoredValue) : array();
        return $Value;
    }

    # ---- PRIVATE INTERFACE -------------------------------------------------

    protected $AudienceCache;

    /**
    * Get/set when the rule was last checked.
    * @param string $NewValue Check timestamp, in format understood
    *       by strtotime().  (OPTIONAL)
    * @return string Previous check timestamp, as a Unix timestamp.
    */
    protected function LastChecked($NewValue = NULL)
    {
        $NewStoredValue = ($NewValue === NULL) ? DB_NOVALUE
                : date("Y-m-d H:i:s", strtotime($NewValue));
        $StoredValue = $this->UpdateValue("LastChecked", $NewStoredValue);
        return strtotime($StoredValue);
    }

    /**
    * Perform search for items matching rule search parameters.
    * @param object $SearchParams Parameters for search.
    * @return array Array of arrays of IDs of items found, indexed (at the
    *       top level) by user ID.
    */
    protected function Search($SearchParams)
    {
        # perform search
        $SEngine = new SPTSearchEngine();
        $RawResults = $SEngine->Search($SearchParams);
        $ItemIds = array_keys($RawResults);

        # pull out the privset used to narrow resource visibility
        if ($this->Action() == self::ACTION_SENDEMAIL)
        {
            $ActionParams = $this->ActionParameters();
            $PrivSet = new PrivilegeSet($ActionParams["Privileges"]);
        }
        else
        {
            $PrivSet = new PrivilegeSet();
        }

        $UFactory = new CWUserFactory();

        # return the visible results that satisfy our Privset for each user
        # (if the privset was empty, we still want to call FUTMR() to
        # convert our array(ItemIds) to array(UserId =>
        # array(ItemIds)), and we'll still want to filter each of
        # those for visibility with FNVRFPUL)
        return CWUserFactory::FilterNonViewableResourcesFromPerUserLists(
            $UFactory->FindUsersThatMeetRequirements($PrivSet, $ItemIds));
    }

    /**
    * Perform action: send email using set template.
    * @param array $ActionParams Additional parameters for action.
    * @param array $ItemLists Array of arrays of IDs of items found, indexed
    *       (at the top level) by user ID.
    */
    protected function PerformAction_SendEmail($ActionParams, $ItemLists)
    {
        # filter out users/items for which email was previously sent
        $LastItemIds = $this->LastMatchingIds();
        foreach ($ItemLists as $UserId => $ItemIds)
        {
            if (isset($LastItemIds[$UserId]))
            {
                $ItemIds = array_diff($ItemIds, $LastItemIds[$UserId]);
            }

            if (count($ItemIds))
            {
                $UserItemLists[$UserId] = $ItemIds;
            }
        }

        # if users were found
        if (isset($UserItemLists))
        {
            # retrieve mailer plugin
            $Mailer = $GLOBALS["G_PluginManager"]->GetPlugin("Mailer");

            # set up extra email substitutions
            $ExtraValues = array(
                    "SEARCHCRITERIA" =>
                            $this->SearchParameters()->TextDescription(),
                    );

            # for each user
            foreach ($UserItemLists as $UserId => $Items)
            {
                # set up user-specific extra email substitutions
                $ExtraValues["TOTALNEWMATCHES"] = count($Items);

                # send email to user
                $Mailer->SendEmail($ActionParams["Template"],
                        $UserId, $Items, $ExtraValues,
                        $ActionParams["ConfirmBeforeSending"]);
            }
        }
    }

    /**
    * Set the database access values (table name, ID column name, name column
    * name) for specified class.  This may be overridden in a child class, if
    * different values are needed.
    * @param string $ClassName Class to set values for.
    */
    static protected function SetDatabaseAccessValues($ClassName)
    {
        if (!isset(self::$ItemIdColumnNames[$ClassName]))
        {
            self::$ItemIdColumnNames[$ClassName] = "RuleId";
            self::$ItemNameColumnNames[$ClassName] = "Name";
            self::$ItemTableNames[$ClassName] = "Rules_Rules";
        }
    }
}
