<?PHP
#
#   FILE:  FieldRedactor.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Allows values for a specified field to be replaced with specified text when
* displayed if the user is not logged in.
*/
class FieldRedactor extends Plugin
{

    /**
    * Register information about this plugin.
    */
    public function Register()
    {
        $this->Name = "Field Redactor";
        $this->Version = "1.0.1";
        $this->Description = "Allows values for a specified field to be"
                ." replaced with specified text when displayed, when user"
                ." is not logged in.";
        $this->Author = "Internet Scout";
        $this->Url = "http://scout.wisc.edu/cwis/";
        $this->Email = "scout@scout.wisc.edu";
        $this->Requires = array(
                "CWISCore" => "2.2.2");
        $this->EnabledByDefault = FALSE;

        $this->CfgSetup["FieldsToFilter"] = array(
                "Type" => "MetadataField",
                "Label" => "Field to Filter",
                "Help" => "Field to replace when being displayed.",
                "AllowMultiple" => TRUE
                );
        $this->CfgSetup["ReplacementText"] = array(
                "Type" => "Paragraph",
                "Label" => "Replacement Text",
                "Help" => "Text to display in place of field value.",
                "Columns" => 60
                );
        $this->CfgSetup["ResourceRatingText"] = array(
                "Type" => "Paragraph",
                "Label" => "Cumulative Rating Replacement Text",
                "Help" => "HTML Text to display in place of resource cumulative rating.",
                "Columns" => 60,
                "ValidateFunction" => function($FieldName, $FieldVal) {
                    if (is_numeric($FieldVal))
                    {
                        return "Replacement text for cumulative rating"
                                ." cannot be a number.\n";
                    }
                    return NULL;
                }
                );
    }

    /**
    * Initialize default settings.
    */
    public function Install()
    {
        $Schema = new MetadataSchema;
        $FieldId = $Schema->StdNameToFieldMapping("Url");
        if ($FieldId !== NULL)
        {
            $this->ConfigSetting("FieldsToFilter", array($FieldId));
            $this->ConfigSetting("ReplacementText",
                    "You must be logged in to access all resource information.");
        }
    }

    /**
    * Hook the events into the application framework.
    * @return Returns an array of events to be hooked into the application
    *      framework.
    */
    public function HookEvents()
    {
        return array(
                "EVENT_FIELD_DISPLAY_FILTER" => "FilterValues",
                );
    }

    /**
    * Redact field values when they are displayed.
    * @param MetadataField $Field Metadata field being displayed.
    * @param Resource $Resource Resource being displayed.
    * @param mixed $Value Value being displayed.
    * @return Returns an array containing the field, resource, and the value,
    *      which might be redacted if it meets the criteria.
    */
    public function FilterValues($Field, $Resource, $Value)
    {
        $FieldsToFilter = $this->ConfigSetting("FieldsToFilter");
        $ResourceRatingText = $this->ConfigSetting("ResourceRatingText");

        if (in_array($Field->Id(), $FieldsToFilter)
            && !$GLOBALS["G_User"]->IsLoggedIn()
            && $this->ShouldReplaceValue($Value))
        {
            # do special filtering for cumulative rating field
            if ($Field->Name() == "Cumulative Rating" && strlen($ResourceRatingText))
            {
                $Value = $ResourceRatingText;
            }
            else
            {
                $Value = $this->ConfigSetting("ReplacementText");
            }
            # check if this is called from SearchResults for search facets
            if ($GLOBALS["AF"]->GetPageName() == "SearchResults")
            {
                $Caller = debug_backtrace()[2];
                if (strpos($Caller["file"], "SearchFacetUI") !== FALSE
                        && $Caller["function"] == "FilterFieldValue")
                {
                    $Value = NULL;
                }
            }
        }

        return array(
                "Field" => $Field,
                "Resource" => $Resource,
                "Value" => $Value);
    }

    /**
    * Determine if the value should be replaced. It shouldn't be replaced if
    * the value is empty or not set.
    * @param mixed $Value Value.
    * @return TRUE if the value should be replaced or FALSE otherwise.
    */
    protected function ShouldReplaceValue($Value)
    {
        # if a scalar and has a length
        if (is_scalar($Value) && strlen($Value) > 0)
        {
            return TRUE;
        }

        # not an empty array
        if (is_array($Value) && count($Value) > 0)
        {
            return TRUE;
        }

        # isn't null
        if (!is_null($Value))
        {
            return TRUE;
        }

        return FALSE;
    }
}