<?PHP
#
#   FILE:  JsonHelper.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2002-2013 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Convenience class for standardizing JSON responses, making it easier to export
* primitive data types to JSON format, and printing JSON responses.
*/
class JsonHelper
{

    /**
    * Object constructor.
    */
    public function __construct()
    {
        $this->Data = array();
        $this->Warnings = array();
    }

    /**
    * Add a datum identified by a key to export in the JSON response.
    * @param string $Key Key used to identify the datum.
    * @param mixed $Value Datum to export.
    */
    public function AddDatum($Key, $Value)
    {
        $this->Data[$Key] = $Value;
    }

    /**
    * Add a warning message to export in the JSON response. Warnings are for
    * issues that might be problematic but won't interrupt execution.
    * @param string $Message Warning message to export.
    * @see Error()
    * @see Success()
    */
    public function AddWarning($Message)
    {
        $this->Warnings[] = strval($Message);
    }

    /**
    * Add an error message to export in the JSON response and then send the
    * response. A possible use of this method is to output and export a message
    * about a missing parameter to a callback.
    * @param string $Message Error message to send.
    * @see Success()
    */
    public function Error($Message)
    {
        $this->SendResult($this->GenerateResult("ERROR", $Message));
    }

    /**
    * Signal that the callback was successful and optionally set a message.
    * @param string $Message Message to export. This parameter is optional.
    * @see Error()
    */
    public function Success($Message="")
    {
        $this->SendResult($this->GenerateResult("OK", $Message));
    }

    private $Data;
    private $Warnings;

    /**
    * Export the data and messages. This sets the Content-Type header and prints
    * the JSON response.
    * @param array $Result Data to export. The data will be converted to JSON.
    */
    private function SendResult(array $Result)
    {
        header("Content-Type: application/json; charset="
            .$GLOBALS["G_SysConfig"]->DefaultCharacterSet(), TRUE);
        $this->PrintArrayToJson($Result);
    }

    /**
    * Generate standard result data based on a final state and message.
    * @param string $State State that the result data is in, e.g., "ERROR" or
    *      "OK"
    * @param string $Message Message to include in the standard result data.
    *     This parameter is optional.
    * @return Returns an array of standard results data.
    */
    private function GenerateResult($State, $Message)
    {
        return array(
            "data" => $this->Data,
            "status" => array(
                "state" => strval($State),
                "message" => strval($Message),
                "numWarnings" => count($this->Warnings),
                "warnings" => $this->Warnings));
    }

    /**
    * Print an array of data in JSON format.
    * @param array $Array Array to print in JSON format.
    */
    private function PrintArrayToJson(array $Array)
    {
        # variables needed for printing commas if necessary
        $Offset = 0;
        $Count = count($Array);

        # determine whether or not we have a true array or a hash map
        $TrueArray = TRUE;
        $ArrayCount = count($Array);
        for ($i = 0, reset($Array); $i < $ArrayCount; $i++, next($Array))
        {
            if (key($Array) !== $i)
            {
                $TrueArray = FALSE;
                break;
            }
        }

        # opening bracket
        print ($TrueArray) ? "[" : "{";

        # print each member
        foreach ($Array as $key => $value)
        {
            # replacements so we can escape strings and replace smart quotes
            static $Replace = array(
                array("\\", "/", "\n", "\t", "\r", "\b", "\f", '"', "",
                        "", "", "", ""),
                array('\\\\', '\\/', '\\n', '\\t', '\\r', '\\b', '\\f', '\"',
                        "'", "'", '\"', '\"', '-'));

            # print key if a hash map
            if (!$TrueArray)
            {
                # escape, remove smart quotes, and print the key
                print '"'.str_replace($Replace[0], $Replace[1], $key).'":';
            }

            # scalar values (int, float, string, or boolean)
            if (is_scalar($value))
            {
                # numeric (i.e., float, int, or float/int string)
                if (is_numeric($value))
                {
                    print $value;
                }

                # string
                else if (is_string($value))
                {
                    # escape, remove smart quotes, and print the value
                    print '"'.str_replace($Replace[0], $Replace[1], $value).'"';
                }

                # boolean true
                else if ($value === TRUE)
                {
                    print "true";
                }

                # boolean false
                else if ($value === FALSE)
                {
                    print "false";
                }
            }

            # recur if the value is an array
            else if (is_array($value))
            {
                $this->PrintArrayToJson($value);
            }

            # null
            else if (is_null($value))
            {
                print "null";
            }

            # object, just print the name and don't possibly expose secret details
            else if (is_object($value))
            {
                print '"object('.get_class($value).')"';
            }

            # resource, just print the name and don't possibly expose secret details
            else
            {
                print '"resource('.get_resource_type($value).')"';
            }

            # print comma if necessary
            if (++$Offset < $Count) {  print ",";  }
        }

        # closing bracket
        print ($TrueArray) ? "]" : "}";
    }
}
