<?PHP
#
#   FILE:  FormUI.php
#
#   Part of the Collection Workflow Integration System (CWIS)
#   Copyright 2016-2017 Edward Almasy and Internet Scout Research Group
#   http://scout.wisc.edu/cwis/
#

/**
* Child class (covering presentation elements only) supplying a standard user
* interface for presenting and working with HTML forms.
*
* NOTE:  If any fields of type FTYPE_PRIVILEGE are included in the form, the
*   enclosing <form> must have the class "priv-form" to work correctly.  (This
*   is a requirement of the PrivilegeEditingUI class.)
*
* USAGE EXAMPLE
*
* This implements a form for editing widget attributes, which are encapsulated
* within a "Widget" class, including photographs of widgets.  The photos are
* displayed in a separate form section, below the other attributes.  If the
* variable $DoNotEditName evaluates to TRUE, the name attribute of the widget
* will be displayed but not be editable.  The same validation function can be
* used for multiple form fields by checking the value of $FieldName inside the
* function to decide how to validate the incoming value(s).
*
* @code
*
*     function MyValidateFunc($FieldName, $FieldValues, $AllFieldValues, $WidgetId)
*     {
*         if ($ValueLooksOkay)
*         {
*             return NULL;
*         }
*         else
*         {
*             return "An informative message about why the value(s) were invalid.";
*         }
*     }
*     $FormFields = array(
*             "WidgetName" => array(
*                     "Type" => FormUI::FTYPE_TEXT,
*                     "Label" => "Widget Name",
*                     "Placeholder" => "(name of widget goes here)",
*                     "Help" => "Name to use for widget.",
*                     "ValidateFunction" => "MyValidateFunc",
*                     "Required" => TRUE,
*                     ),
*             "WidgetDescription" => array(
*                     "Type" => FormUI::FTYPE_PARAGRAPH,
*                     "Label" => "Description",
*                     "UseWYSIWYG" => TRUE,
*                     "Help" => "Lurid description of widget.",
*                     ),
*             "WidgetType" => array(
*                     "Type" => FormUI::FTYPE_OPTION,
*                     "Label" => "Type of Widget",
*                     "Help" => "The type of widget.",
*                     "Options" => array(
*                             WidgetClass::WT_NONE => "--",
*                             WidgetClass::WT_FIRST => "First Type",
*                             WidgetClass::WT_SECOND => "Second Type",
*                             ),
*                     ),
*             "WidgetCount" => array(
*                     "Type" => FormUI::FTYPE_NUMBER,
*                     "Label" => "Number of Widgets",
*                     "MinVal" => 1,
*                     "MaxVal" => 100,
*                     "Required" => TRUE,
*                     "Value" => $Widget->Count(),
*                     ),
*             "PhotoSectionHeader" => array(
*                     "Type" => FormUI::FTYPE_HEADING,
*                     "Label" => "Other Widget Info",
*                     ),
*             "WidgetPhotos" => array(
*                     "Type" => FormUI::FTYPE_IMAGE,
*                     "Label" => "Widget Images",
*                     "Help" => "Photos of widgets.",
*                     "AllowMultiple" => TRUE,
*                     "InsertIntoField" => "WidgetDescription",
*                     "MaxWidth" => 1000,
*                     "MaxHeight" => 1000,
*                     "MaxPreviewWidth" => 100,
*                     "MaxPreviewHeight" => 100,
*                     "MaxThumbnailWidth" => 50,
*                     "MaxThumbnailHeight" => 50,
*                     ),
*             );
*     if ($DoNotEditName)
*     {
*         $FormFields["WidgetName"]["ReadOnly"] = TRUE;
*     }
*     $FormValues = array(
*             "WidgetName" => $Widget->Name(),
*             "WidgetDescription" => $Widget->Description(),
*             "WidgetType" => $Widget->Type(),
*             "WidgetPhotos" => array_keys(
*                     $Widget->PhotosWithIdsForArrayIndex());
*             );
*     $H_FormUI = new FormUI($FormFields, $FormValues);
*     $H_FormUI->AddValidationParameters($Widget->Id());
*
*     switch ($ButtonPushed)
*     {
*         case "Save":
*             if ($H_FormUI->ValidateFieldInput() == 0)
*             {
*                 $NewValues = $H_FormUI->GetNewValuesFromForm();
*                 $Widget->Name($NewValues["WidgetName"]);
*                 $Widget->Description($NewValues["WidgetDescription"]);
*                 $Widget->Count($NewValues["WidgetCount"]);
*                 $GLOBALS["AF"]->SetJumpToPage("BackToWhereverWhenDone");
*             }
*             break;
*         case "Upload":
*             $H_FormUI->HandleUploads();
*             break;
*         case "Delete":
*             $H_FormUI->HandleDeletes();
*             break;
*     }
*
*     FormUI::DisplayErrorBlock();
*     (start form)
*     $H_FormUI->DisplayFormTable();
*     (add submit buttons)
*     (end form)
*
* @endcode
*/
class FormUI extends FormUI_Base
{

    # ---- PUBLIC INTERFACE --------------------------------------------------

    /**
    * Display HTML table for form.
    * @param string $TableId CSS ID for table element.  (OPTIONAL)
    * @param string $TableStyle CSS styles for table element.  (OPTIONAL)
    * @param string $TableCssClass Additional CSS class for table element. (OPTIONAL)
    */
    public function DisplayFormTable($TableId = NULL, $TableStyle = NULL,
            $TableCssClass = NULL)
    {
        # display nothing if there are no fields
        if (!count($this->FieldParams)) {  return;  }

        # check whether table should be split into sections
        $TableIsSectioned = FALSE;
        foreach ($this->FieldParams as $Name => $Params)
        {
            if ($Params["Type"] == self::FTYPE_HEADING)
                    {  $TableIsSectioned = TRUE;  }
        }

        # begin table
        // @codingStandardsIgnoreStart
        ?><table class="cw-table cw-table-fullsize cw-table-sideheaders cw-table-padded cw-table-striped<?PHP
        if ($TableIsSectioned) {  print(" cw-table-sectioned");  }
        if (!is_null($TableCssClass)) {  print(" ".$TableCssClass);  }
        ?> cw-content-sysconfigtable"<?PHP
        if ($TableId) {  print(" id=\"".$TableId."\"");  }
        if ($TableStyle) {  print(" style=\"".$TableStyle."\"");  }
        ?>>
        <tbody><?PHP
        // @codingStandardsIgnoreEnd

        # for each field
        foreach ($this->FieldParams as $Name => $Params)
        {
            # generate name for field
            $FormFieldName = $this->GetFormFieldName($Name);

            # if field is actually a section heading
            if ($Params["Type"] == self::FTYPE_HEADING)
            {
                # split table section and display heading
                if (isset($HeadingAlreadyDisplayed)) {  print("</tbody><tbody>");  }
                ?><tr id="section-<?= $FormFieldName
                        ?>"><th colspan="3" scope="rowspan"><?=
                $Params["Label"] ?></th></tr><?PHP
                $HeadingAlreadyDisplayed = TRUE;
            }
            else
            {
                # determine if row may have taller content
                $ShortRowFieldTypes = array(
                        self::FTYPE_FLAG,
                        self::FTYPE_METADATAFIELD,
                        self::FTYPE_NUMBER,
                        self::FTYPE_PASSWORD,
                        self::FTYPE_TEXT,
                        self::FTYPE_URL,
                        self::FTYPE_USER,
                        );
                $IsTallRow =
                    ($Params["Type"] == self::FTYPE_QUICKSEARCH) ||
                    (!isset($Params["Units"])
                        && !in_array($Params["Type"], $ShortRowFieldTypes)
                        && (($Params["Type"] != self::FTYPE_OPTION)
                                || (isset($Params["Rows"])
                                        && ($Params["Rows"] > 1))) );

                # load up value(s) to go into field
                $Value = $this->GetFieldValue($Name);

                # set up CSS classes for table row
                $RowClass = "cw-formui-fieldtype-".strtolower($Params["Type"]);
                if ($Params["Type"] == "MetadataField")
                {
                    $RowClass .= " cw-formui-schemaid-"
                            .GetArrayValue($Params, "SchemaId",
                                    MetadataSchema::SCHEMAID_DEFAULT);
                }
                $RowClass .= $IsTallRow ? " cw-content-tallrow" : "";
                $RowClassAttrib = ' class="'.$RowClass.'"';

                # set up CSS classes for row header cell
                $HeaderClass = $IsTallRow ? "cw-content-tallrow-th" : "";
                $HeaderClassAttrib = strlen($HeaderClass)
                        ? ' class="'.$HeaderClass.'"' : "";

                # set up CSS classes for row label
                $LabelClass = "cw-form-pseudolabel"
                        .(isset(self::$ErrorMessages[$Name])
                        ? " cw-form-error" : "");

                # set up min/max note if applicable
                unset($RangeNotePieces);
                if (isset($Params["MinVal"]))
                {
                    $RangeNotePieces[] = "Minimum: <i>".$Params["MinVal"]."</i>";
                }
                if (isset($Params["MaxVal"]))
                {
                    $RangeNotePieces[] = "Maximum: <i>".$Params["MaxVal"]."</i>";
                }
                if (isset($Params["RecVal"]))
                {
                    $RangeNotePieces[] = "Recommended: <i>".$Params["RecVal"]."</i>";
                }
                if (isset($RangeNotePieces))
                {
                    $RangeNote = "(".implode(", ", $RangeNotePieces).")";
                }

                // @codingStandardsIgnoreStart
                ?>
                <tr<?= ($IsTallRow ? " valign=\"top\"" : "").$RowClassAttrib
                            ?> id="row-<?= $FormFieldName ?>">
                    <th<?= $HeaderClassAttrib ?>>
                        <label for="<?=  $FormFieldName
                                ?>" class="<?=  $LabelClass  ?>"><?=
                                $Params["Label"]  ?></label>
                    </th>
                    <td <?PHP  if (!isset($Params["Help"]) && !isset($RangeNotePieces)) {
                                    print "colspan=\"2\"";  }  ?>><?PHP
                            $this->DisplayFormField(
                                    $Name, $Value, $Params);  ?></td>
                    <?PHP  if (isset($Params["Help"])) {  ?>
                    <td class="cw-content-help-cell"><?= $Params["Help"] ?></td>
                    <?PHP  } elseif (isset($RangeNotePieces)) {  ?>
                    <td class="cw-content-help-cell"><?= $RangeNote ?></td>
                    <?PHP  }  ?>
                </tr>
                <?PHP
            }
        }

        # end table
        ?></tbody>
        </table><?PHP

        # add any hidden form fields
        print $this->GetHiddenFieldsHtml();

        # add any needed JavaScript for toggled fields
        $this->PrintFieldHidingJavascript();

        # pull in WYSIWYG editor setup if needed
        if ($this->UsingWysiwygEditor)
        {
            require_once($GLOBALS["AF"]->GUIFile("CKEditorSetup.php"));
        }
    }
    // @codingStandardsIgnoreEnd

    /**
    * Display HTML block with error messages (if any).
    */
    public static function DisplayErrorBlock()
    {
        if (count(self::$ErrorMessages))
        {
            print "<ul class=\"cw-form-error\">\n";
            $DisplayedMsgs = array();
            foreach (self::$ErrorMessages as $Field => $Msgs)
            {
                foreach ($Msgs as $Msg)
                {
                    if (!in_array($Msg, $DisplayedMsgs))
                    {
                        print "<li>".$Msg."</li>\n";
                        $DisplayedMsgs[] = $Msg;
                    }
                }
            }
            print "</ul>\n";
        }
    }

    /**
    * Handle image deletion, removing deleted images from text fields
    * where they may have been inserted.
    */
    public function HandleDeletes()
    {
        parent::HandleDeletes();

        $TextFieldsToCheck = array();

        # check for text fields that may contain images
        foreach ($this->FieldParams as $Name => $Params)
        {
            if (isset($Params["InsertIntoField"])
                    && (($Params["Type"] == self::FTYPE_FILE)
                            || ($Params["Type"] == self::FTYPE_IMAGE)))
            {
                $TextFieldsToCheck[] = $Params["InsertIntoField"];
            }
        }

        # load images to check
        foreach ($this->DeletedImages as $ImageId)
        {
            $Images[$ImageId] = new SPTImage($ImageId);
        }

        # load files to check
        foreach ($this->DeletedFiles as $FileId)
        {
            $Files[$FileId] = new File($FileId);
        }

        # for each text field potentially containing references to deleted items
        foreach ($TextFieldsToCheck as $FieldName)
        {
            # get HTML form field name for field
            $FormFieldName = $this->GetFormFieldName($FieldName);

            # for each deleted image
            foreach ($this->DeletedImages as $ImageId)
            {
                # strip out any tags referencing that image from field content
                $_POST[$FormFieldName] = preg_replace(
                        "%<img [^>]*src=\""
                                .$Images[$ImageId]->PreviewUrl()."\"[^>]*>%",
                        "",
                        $this->GetFieldValue($FieldName));
            }

            # for each deleted file
            foreach ($this->DeletedFiles as $FileId)
            {
                # strip out any tags we inserted that reference that file
                $FileLink = $Files[$FileId]->GetLink();
                $_POST[$FormFieldName] = preg_replace(
                        "%<a [^>]*href=\""
                                .preg_quote(htmlspecialchars($FileLink), '%')
                                ."\"[^>]*>(.*?)</a>%",
                        "\1",
                        $this->GetFieldValue($FieldName));
            }
        }
    }



    # ---- PRIVATE INTERFACE -------------------------------------------------

    protected $UsingWysiwygEditor = FALSE;

    /**
    * FTYPE_OPTION fields with this many or fewer options will display as
    * radio buttons by default.
    */
    const OPTION_RADIO_BUTTON_THRESHOLD = 2;

    /**
    * Display HTML form field for specified field.
    * @param string $Name Field name.
    * @param mixed $Value Current value for field.
    * @param array $Params Field parameters.
    */
    protected function DisplayFormField($Name, $Value, $Params)
    {
        $FieldName = $this->GetFormFieldName($Name,
                ($Params["Type"] != self::FTYPE_PRIVILEGES));

        switch ($Params["Type"])
        {
            case self::FTYPE_TEXT:
            case self::FTYPE_NUMBER:
            case self::FTYPE_URL:
            case self::FTYPE_PASSWORD:
                $DefaultSize = ($Params["Type"] == self::FTYPE_NUMBER) ? 6 : 40;
                $DefaultMaxLen = ($Params["Type"] == self::FTYPE_NUMBER) ? 12 : 80;
                $Size = isset($Params["Size"]) ? $Params["Size"]
                        : (isset($Params["MaxVal"])
                            ? (strlen(intval($Params["MaxVal"]) + 1))
                            : $DefaultSize);
                $MaxLen = isset($Params["MaxLength"]) ? $Params["MaxLength"]
                        : (isset($Params["MaxVal"])
                            ? (strlen(intval($Params["MaxVal"]) + 3))
                            : $DefaultMaxLen);
                $Placeholder = isset($Params["Placeholder"])
                        ? $Params["Placeholder"]
                        : "(".strtolower($Params["Label"]).")";
                $InputType = ($Params["Type"] == self::FTYPE_PASSWORD)
                        ? "password" : "text";
                print('<input type="'.$InputType.'" size="'.$Size.'" maxlength="'
                        .$MaxLen.'" id="'.$FieldName.'" name="'.$FieldName.'"'
                        .' value="'.htmlspecialchars($Value).'"'
                        .' placeholder=" '.htmlspecialchars($Placeholder).'"'
                        .($Params["ReadOnly"] ? " readonly" : "").' />');
                break;

            case self::FTYPE_PARAGRAPH:
                $Rows = isset($Params["Rows"]) ? $Params["Rows"]
                        : (isset($Params["Height"]) ? $Params["Height"] : 4);
                $Columns = isset($Params["Columns"]) ? $Params["Columns"]
                        : (isset($Params["Width"]) ? $Params["Width"] : 40);
                print('<textarea rows="'.$Rows.'" cols="'.$Columns
                        .'" id="'.$FieldName.'" name="'.$FieldName.'"'
                        .($Params["ReadOnly"] ? " readonly" : "")
                        .($Params["UseWYSIWYG"] ? ' class="ckeditor"' : "").'>'
                        .htmlspecialchars($Value)
                        .'</textarea>');
                if ($Params["UseWYSIWYG"])
                {
                    $this->UsingWysiwygEditor = TRUE;
                }
                break;

            case self::FTYPE_FLAG:
                if (array_key_exists("OnLabel", $Params)
                        && array_key_exists("OffLabel", $Params))
                {
                    print('<input type="radio" id="'.$FieldName.'On" name="'
                            .$FieldName.'" value="1"'
                            .($Value ? ' checked' : '')
                            .($Params["ReadOnly"] ? ' disabled' : '')
                            .' /> <label for="'.$FieldName.'On">'.$Params["OnLabel"]
                            ."</label>\n");
                    print('<input type="radio" id="'.$FieldName.'Off" name="'
                            .$FieldName.'" value="0"'
                            .($Value ? '' : ' checked')
                            .($Params["ReadOnly"] ? ' disabled' : '')
                            .' /> <label for="'.$FieldName.'Off">'.$Params["OffLabel"]
                            ."</label>\n");
                }
                else
                {
                    print('<input type="checkbox" id="'.$FieldName.'" name="'
                            .$FieldName.'" '
                            .($Value ? ' checked' : '')
                            .($Params["ReadOnly"] ? ' disabled' : '')
                            ." />\n");
                }
                break;

            case self::FTYPE_OPTION:
                if ($this->IsRadioButtonField($Name))
                {
                    $OptList = new HtmlRadioButtonSet(
                            $FieldName, $Params["Options"], $Value);
                }
                else
                {
                    $OptList = new HtmlOptionList(
                            $FieldName, $Params["Options"], $Value);
                    $OptList->MultipleAllowed($Params["AllowMultiple"]);
                    $OptList->Size(isset($Params["Rows"]) ? $Params["Rows"] : 1);
                }
                $OptList->Disabled($Params["ReadOnly"]);
                $OptList->PrintHtml();
                break;

            case self::FTYPE_METADATAFIELD:
                $FieldTypes = GetArrayValue($Params, "FieldTypes");
                $SchemaId = GetArrayValue($Params, "SchemaId",
                        MetadataSchema::SCHEMAID_DEFAULT);
                $Schema = new MetadataSchema($SchemaId);
                print $Schema->GetFieldsAsOptionList(
                        $FieldName, $FieldTypes, $Value,
                        !$Params["AllowMultiple"] && !$Params["Required"],
                        NULL, $Params["AllowMultiple"], $Params["ReadOnly"]);
                break;

            case self::FTYPE_PRIVILEGES:
                # (convert legacy previously-stored values if necessary)
                if (is_array($Value))
                {
                    $PrivSet = new PrivilegeSet();
                    $PrivSet->AddPrivilege($Value);
                    $Value = $PrivSet;
                }

                $Schemas = GetArrayValue($Params, "Schemas");
                $MFields = GetArrayValue($Params, "MetadataFields", array());
                $PEditor = new PrivilegeEditingUI($Schemas, $MFields);
                $PEditor->DisplaySet($FieldName, $Value);
                break;

            case self::FTYPE_SEARCHPARAMS:
                $SPEditor = new SearchParameterSetEditingUI($FieldName, $Value);

                if (isset($Params["MaxFieldLabelLength"]))
                {
                    $SPEditor->MaxFieldLabelLength($Params["MaxFieldLabelLength"]);
                }
                if (isset($Params["MaxValueLabelLength"]))
                {
                    $SPEditor->MaxValueLabelLength($Params["MaxValueLabelLength"]);
                }

                $SPEditor->DisplayAsTable();
                break;

            case self::FTYPE_USER:
            case self::FTYPE_QUICKSEARCH:
                if (is_null($Value))
                {
                    $Value = [];
                }

                # set up some helpers that abstract over the
                # differences between a USER and a QUICKSEARCH field
                if ($Params["Type"] == self::FTYPE_USER)
                {
                    $Search = "UserSearch";
                    $AllowMultiple = $Params["AllowMultiple"];
                    $UFactory = new CWUserFactory();
                    $NameFn = function($Key, $Val) use ($UFactory) {
                        return $UFactory->UserExists($Val) ?
                                (new CWUser($Val))->Name() : "" ;
                    };
                    $IdFn = function($Key, $Val) {
                        return $Val;
                    };
                }
                else
                {
                    $MField = new MetadataField($Params["Field"]);
                    $Search = $Params["Field"];
                    $AllowMultiple = $MField->AllowMultlple();
                    $NameFn = function($Key, $Val) use ($MField) {
                        if ($MField->Type() == MetadataSchema::MDFTYPE_REFERENCE)
                        {
                            $Resource = new Resource($Key);
                            return $Resource->GetMapped("Title");
                        }
                        else
                        {
                            return $Val;
                        }
                    };
                    $IdFn = function($Key, $Val) {
                        return $Key;
                    };
                }

                # filter out empty incoming values
                $Value = array_filter(
                    $Value, function($x) {
                        return strlen($x)>0;
                    });

                if (count($Value))
                {
                    # iterate over incoming values
                    foreach ($Value as $Key => $Val)
                    {
                        # pull out the corresponding name/id
                        $VName = $NameFn($Key, $Val);
                        $VId = $IdFn($Key, $Val);

                        # print UI elements
                        if ($Params["ReadOnly"])
                        {
                            print "<p>".defaulthtmlentities($VName)."</p>";
                        }
                        else
                        {
                            QuickSearchHelper::PrintQuickSearchField(
                                $Search,
                                $VId,
                                defaulthtmlentities($VName),
                                FALSE,
                                $FieldName);
                        }
                    }
                }

                if (!$Params["ReadOnly"])
                {
                    # display a blank row for adding more values
                    # when we have no values or when we allow more
                    if (count($Value)==0 || $AllowMultiple)
                    {
                        QuickSearchHelper::PrintQuickSearchField(
                            $Search,
                            "",
                            "",
                            $AllowMultiple,
                            $FieldName);
                    }
                }
                break;

            case self::FTYPE_FILE:
                $this->DisplayFileField($FieldName, $Value, $Params);
                break;

            case self::FTYPE_IMAGE:
                $this->DisplayImageField($FieldName, $Value, $Params);
                break;
        }

        if (isset($Params["Units"]))
        {
            ?>&nbsp;<span><?PHP
            print $Params["Units"];
            ?></span><?PHP
        }
    }

    /**
    * Display image form field for specified field.
    * @param string $FieldName Field name.
    * @param mixed $Value Current value for field.
    * @param array $Params Field parameters.
    */
    protected function DisplayImageField($FieldName, $Value, $Params)
    {
        # normalize incoming value
        $Images = is_array($Value) ? $Value
                : (($Value === NULL) ? array() : array($Value));

        # begin value table
        print '<table border="0" cellspacing="0" cellpadding="0" width="100%">';

        # for each incoming value
        $ImagesDisplayed = 0;
        $InsertButtonHtml = "";
        foreach ($Images as $Image)
        {
            # skip if image is a placeholder to indicate no images for field
            if ($Image == self::NO_VALUE_FOR_FIELD)
            {
                continue;
            }

            # load up image object if ID supplied
            if (is_numeric($Image))
            {
                $Image = new SPTImage($Image);
            }

            # skip image if it has been deleted
            if (in_array($Image->Id(), $this->DeletedImages))
            {
                continue;
            }

            # load various image attributes for use in HTML
            $ImageUrl = defaulthtmlentities($Image->ThumbnailUrl());
            $ImageId = $Image->Id();
            $ImageAltTextFieldName = $FieldName."_AltText_".$ImageId;
            $ImageAltText = defaulthtmlentities(
                    isset($_POST[$ImageAltTextFieldName])
                    ? $_POST[$ImageAltTextFieldName]
                    : $Image->AltText());

            $DeleteFieldName = $this->GetFormFieldName("ImageToDelete");

            # build up HTML for any insert buttons
            if (isset($Params["InsertIntoField"]))
            {
                $InsertField = $this->GetFormFieldName($Params["InsertIntoField"]);
                $InsertRightCommand = defaulthtmlentities(
                        "CKEDITOR.instances['".$InsertField
                                ."'].insertHtml("
                        ."'<img src=\"".$Image->PreviewUrl()."\" alt=\""
                        .htmlspecialchars($Image->AltText())
                        ."\" class=\"cw-formui-image-right\" />');");
                $InsertLeftCommand = defaulthtmlentities(
                        "CKEDITOR.instances['".$InsertField
                                ."'].insertHtml("
                        ."'<img src=\"".$Image->PreviewUrl()."\" alt=\""
                        .htmlspecialchars($Image->AltText())
                        ."\" class=\"cw-formui-image-left\" />');");
                $InsertButtonHtml = '<button type="button" onclick="'
                        .$InsertLeftCommand.'">Insert-L</button>'
                        .'<button type="button" onclick="'
                        .$InsertRightCommand.'">Insert-R</button>';
            }

            # add table row for image
            ?><tr>
                <td><img src="<?= $ImageUrl ?>"></td>
                <td style="white-space: nowrap;"><label for="<?=
                        $ImageAltTextFieldName ?>" class="cw-form-pseudolabel">
                        Alt Text:</label><input type="text" size="20"
                        maxlength="120" name="<?= $ImageAltTextFieldName ?>"
                        value="<?= $ImageAltText ?>"
                        placeholder=" (alternate text for image)"></td>
                <td><?= $InsertButtonHtml ?><input type="submit" name="Submit"
                        onclick="$('#<?= $DeleteFieldName ?>').val('<?= $ImageId
                        ?>');" value="Delete"></td>
            </tr><?PHP
            $ImagesDisplayed++;

            # add image ID to hidden fields
            $this->HiddenFields[$FieldName."_ID"][] = $Image->Id();

            # add container to hold ID of any image to be deleted
            if (!isset($this->HiddenFields[$DeleteFieldName]))
            {
                $this->HiddenFields[$DeleteFieldName] = "";
            }
        }

        # if no images were displayed and an image entry was skipped
        if (($ImagesDisplayed == 0) && count($Images))
        {
            # add marker to indicate no images to hidden fields
            $this->HiddenFields[$FieldName."_ID"][] = self::NO_VALUE_FOR_FIELD;
        }

        # add table row for new image upload
        if ($Params["AllowMultiple"] || ($ImagesDisplayed == 0))
        {
            $ImageAltTextFieldName = $FieldName."_AltText_NEW";
            ?><tr>
                <td><input type="file" name="<?= $FieldName ?>" /></td>
                <td style="white-space: nowrap;"><label for="<?=
                        $ImageAltTextFieldName ?>" class="cw-form-pseudolabel">
                        Alt Text:</label><input type="text" size="20"
                        maxlength="120" name="<?= $ImageAltTextFieldName ?>"
                        placeholder=" (alternate text for image)"></td>
                <td><input type="submit" name="Submit" value="Upload" /></td>
            </tr><?PHP
        }

        # end value table
        print '</table>';
    }

    /**
    * Display file form field for specified field.
    * @param string $FieldName Field name.
    * @param mixed $Value Current value for field.
    * @param array $Params Field parameters.
    */
    protected function DisplayFileField($FieldName, $Value, $Params)
    {
        # normalize incoming value
        $Files = is_array($Value) ? $Value
                : (($Value === NULL) ? array() : array($Value));

        # begin value table
        print '<table border="0" cellspacing="0" cellpadding="0" width="100%">';

        # for each incoming value
        $FilesDisplayed = 0;
        $InsertButtonHtml = "";
        foreach ($Files as $File)
        {
            # skip if file is a placeholder to indicate no files for field
            if ($File == self::NO_VALUE_FOR_FIELD)
            {
                continue;
            }

            # load up file object if ID supplied
            if (is_numeric($File))
            {
                $File = new File($File);
            }

            # skip file if it has been deleted
            if (in_array($File->Id(), $this->DeletedFiles))
            {
                continue;
            }

            # load various attributes for use in HTML
            $FileId = $File->Id();
            $FileUrl = $File->GetLink();
            $FileName = $File->Name();
            $FileLinkTag = "<a href=\"".htmlspecialchars($FileUrl)
                        ."\" target=\"_blank\">"
                        .htmlspecialchars($FileName)."</a>";
            $DeleteFieldName = $this->GetFormFieldName("FileToDelete");

            # build up HTML for any insert buttons
            if (isset($Params["InsertIntoField"]))
            {
                $InsertField = $this->GetFormFieldName($Params["InsertIntoField"]);
                $InsertCommand = defaulthtmlentities(
                        "CKEDITOR.instances['".$InsertField
                                ."'].insertHtml('".$FileLinkTag."');");
                $InsertButtonHtml = '<button type="button" onclick="'
                        .$InsertCommand.'">Insert</button>';
            }

            # add table row for file
            ?><tr>
                <td><?= $FileLinkTag ?></td>
                <td><?= $InsertButtonHtml ?><input type="submit" name="Submit"
                        onclick="$('#<?= $DeleteFieldName ?>').val('<?= $FileId
                        ?>');" value="Delete"></td>
            </tr><?PHP
            $FilesDisplayed++;

            # add file ID to hidden fields
            $this->HiddenFields[$FieldName."_ID"][] = $FileId;

            # add container to hold ID of any file to be deleted
            if (!isset($this->HiddenFields[$DeleteFieldName]))
            {
                $this->HiddenFields[$DeleteFieldName] = "";
            }
        }

        # if no files were displayed and a file entry was skipped
        if (($FilesDisplayed == 0) && count($Files))
        {
            # add marker to indicate no files to hidden fields
            $this->HiddenFields[$FieldName."_ID"][] = self::NO_VALUE_FOR_FIELD;
        }

        # add table row for new file upload
        if ($Params["AllowMultiple"] || ($FilesDisplayed == 0))
        {
            ?><tr>
                <td><input type="file" name="<?= $FieldName ?>" /></td>
                <td><input type="submit" name="Submit" value="Upload" /></td>
            </tr><?PHP
        }

        # end value table
        print '</table>';
    }

    /**
    * Print any JavaScript required to support toggling display of fields
    * or sections.
    */
    protected function PrintFieldHidingJavascript()
    {
        # for each form field
        foreach ($this->FieldParams as $ToggledField => $Params)
        {
            # if field has togglers (other fields that can toggle this field)
            if (isset($Params["DisplayIf"]))
            {
                # for each toggler
                foreach ($Params["DisplayIf"] as $Toggler => $ToggleValues)
                {
                    # add field to list of fields toggled by this toggler
                    $FieldsToggled[$Toggler][] = $ToggledField;

                    # add values to list of values that toggle this field
                    if (!is_array($ToggleValues))
                            {  $ToggleValues = array($ToggleValues);  }
                    $FieldToggleValues[$ToggledField][$Toggler] = $ToggleValues;
                }
            }
        }

        # if there were fields that toggle other fields
        if (isset($FieldsToggled))
        {
            # start JavaScript code
            ?>
            <script type="text/javascript">
            (function($){
            <?PHP

            # for each toggler
            foreach ($FieldsToggled as $Toggler => $ToggledFields)
            {
                # begin function called when toggler changes
                $TogglerFFName = $this->GetFormFieldName($Toggler);
                ?>
                $("[id^=<?= $TogglerFFName ?>]").change(function(){
                <?PHP

                # for each togglee (field being toggled)
                foreach ($ToggledFields as $ToggledField)
                {
                    # get JavaScript condition for this togglee
                    $ConditionJS = $this->GetFieldToggleConditionJS(
                            $FieldToggleValues[$ToggledField]);

                    # add toggle code for togglee
                    $ToggledFieldFFName = $this->GetFormFieldName($ToggledField);
                    ?>
                    $("#row-<?= $ToggledFieldFFName ?>")[<?=
                            $ConditionJS ?> ? "show" : "hide"]();
                    <?PHP
                }

                # end function for field changing
                ?>
                }).change();
                <?PHP
            }

            # end JavaScript code
            ?>
            }(jQuery));
            </script>
            <?PHP
        }
    }

    /**
    * Get JavaScript snippet that can be used for toggling a form field.
    * @param array $ToggleValues Values that toggle field, with togglers
    *       (form fields to which may contain the values) for the index.
    * @return string JavaScript condition string.
    */
    private function GetFieldToggleConditionJS($ToggleValues)
    {
        # for each toggler
        foreach ($ToggleValues as $Toggler => $ToggleValues)
        {
            # start with fresh subcondition list
            $SubConditions = array();

            # for each toggle value
            $TogglerFFName = $this->GetFormFieldName($Toggler);
            foreach ($ToggleValues as $Value)
            {
                # build subcondition for value
                if ($this->FieldParams[$Toggler]["Type"] == self::FTYPE_FLAG)
                {
                    if ($Value)
                    {
                        $SubConditions[] = "($(\"input[name=".$TogglerFFName
                                ."]\").is(\":checked:visible\"))";
                    }
                    else
                    {
                        $SubConditions[] = "(!$(\"input[name=".$TogglerFFName
                                ."]\").is(\":checked:visible\"))";
                    }
                }
                else
                {
                    if ($this->IsRadioButtonField($Toggler))
                    {
                        $SubConditions[] = "($(\"input[name=".$TogglerFFName
                                ."]:checked\").val() == \"".$Value."\")";
                    }
                    else
                    {
                        $SubConditions[] = "($(\"#".$TogglerFFName
                                ."\").val() == \"".$Value."\")";
                    }
                }
            }

            # assemble subconditions into condition
            if (count($SubConditions) > 1)
            {
                $SubConditionStrings[] = "(".implode(" || ", $SubConditions).")";
            }
            else
            {
                $SubConditionStrings[] = $SubConditions[0];
            }
        }

        # assemble conditions into condition string
        $ConditionString = implode(" && ", $SubConditionStrings);

        return $ConditionString;
    }

    /**
    * Check whether specified field should be displayed as radio buttons.
    * @param string $FieldName Name of field.
    * @return bool TRUE if field should use radio buttons, otherwise FALSE.
    */
    private function IsRadioButtonField($FieldName)
    {
        $Params = $this->FieldParams[$FieldName];
        if ($Params["Type"] == self::FTYPE_OPTION)
        {
            if (isset($Params["RadioButtons"]))
            {
                return $Params["RadioButtons"] ? TRUE : FALSE;
            }
            elseif (isset($Params["AllowMultiple"]) && $Params["AllowMultiple"])
            {
                return FALSE;
            }
            else
            {
                return (count($Params["Options"])
                        <= self::OPTION_RADIO_BUTTON_THRESHOLD) ? TRUE : FALSE;
            }
        }
        else
        {
            return FALSE;
        }
    }
}
