<?PHP

/**
* Print summary containing standard info (title, description, URL)
* about the given Resource.
* @param Resource $Resource Resource object.
* @return string  HTML to print the resource summary for
*      display in a QuickSearch
*/
function GetResourceSummaryForQuicksearch($Resource)
{
    #----- CONFIGURATION ----------------------------------------------

    # maximum length of displayed values (in characters)
    $MaxLengthResourceLink = 60;
    $MaxDescriptionLength = 250;

    #----- SETUP ------------------------------------------------------

    global $G_SysConfig, $G_User;

    # retrieve and format title
    $Schema = new MetadataSchema($Resource->SchemaId());
    $TitleField = $Schema->GetFieldByMappedName("Title");
    if ($Resource->UserCanViewField($G_User, $TitleField))
    {
        $Title = StripXSSThreats($Resource->GetForDisplay($TitleField));
    }

    $UrlLooksValid = TRUE;

    # get URL link (if any)
    $UrlField = $Schema->GetFieldByMappedName("Url");

    if ($UrlField !== NULL &&
        $Resource->UserCanViewField($G_User, $UrlField))
    {
        $Url = $Resource->GetForDisplay($UrlField);

        if (strlen($Url))
        {
            $UrlLink = "index.php?P=GoTo&amp;ID=".$Resource->Id()
                    ."&amp;MF=".$UrlField->Id();
            $RealUrlLink = $Resource->GetForDisplay($UrlField);

            if (!preg_match('/^\s*[a-zA-Z]+:\/\//', $RealUrlLink))
            {
                $UrlLooksValid = FALSE;
            }
        }
    }

    # get file link (if any)
    $FileField = $Schema->GetFieldByMappedName("File");
    if ($FileField !== NULL &&
        $Resource->UserCanViewField($G_User, $FileField))
    {
        $FileList = $Resource->GetForDisplay($FileField, TRUE);
        if (is_array($FileList) && count($FileList))
        {
            $File = array_shift($FileList);
            $FileLink = ApplicationFramework::BaseUrl().$File->GetLink();
        }
    }

    # figure out which link to use (Url or File)
    if (isset($UrlLink) && isset($FileLink))
    {
        $ResourceLink = ($G_SysConfig->Value("PreferredLinkValue") == "FILE")
                ? $FileLink : $UrlLink;
    }
    elseif (isset($UrlLink))
    {
        $ResourceLink = $UrlLink;
    }
    elseif (isset($FileLink))
    {
        $ResourceLink = $FileLink;
    }

    $UrlLooksValid = isset($ResourceLink) && $UrlLooksValid;

    # construct an "<a href" opening tag pointing to our desired link
    if (isset($ResourceLink))
    {
        $ResourceLinkTag = "<a href=\"".$ResourceLink."\" title=\"Go to "
                .(isset($Title) ? htmlspecialchars(strip_tags($Title)) : "Resource")."\""
                .($G_SysConfig->Value("ResourceLaunchesNewWindowEnabled")
                        ? " target=\"_blank\"" : "").">";
    }

    # if we're using the URL value
    if (isset($UrlLink) && ($ResourceLink == $UrlLink))
    {
        # and if it's overly long
        if (strlen($RealUrlLink) > $MaxLengthResourceLink &&
            strlen(strip_tags($RealUrlLink))==strlen($RealUrlLink))
        {
            # truncate it to the desired length
            $DisplayableResourceLink = substr($RealUrlLink, 0,
            $MaxLengthResourceLink)."...";
        }
        else
        {
            # otherwise just use it directly
            $DisplayableResourceLink = $RealUrlLink;
        }
    }

    # get link to full record page
    $FullRecordLink = "index.php?P=FullRecord&amp;ID=".$Resource->Id();
    $FullRecordLinkTag = "<a href=\"".$FullRecordLink."\""
            ." title=\"View More Info for ".(isset($Title)
                    ? htmlspecialchars(strip_tags($Title)) : "Resource")."\">";

    # pull out the mapped description, if one is available
    $DescriptionField = $Schema->GetFieldByMappedName("Description");
    if ($DescriptionField instanceof MetadataField
        && $Resource->UserCanViewField($G_User, $DescriptionField))
    {

        # get stripped and length-limited version of description
        $Description = strip_tags(
        $Resource->GetForDisplay($DescriptionField),
            "<b><i><u><sub><sup><strike><a>");
        $Description = StdLib::NeatlyTruncateString($Description, $MaxDescriptionLength);
    }

    #----- DISPLAY ----------------------------------------------------

    /*
    * At this point these values should be available:
    *   $FullRecordLink - link to full record page
    *   $FullRecordLinkTag - complete <a> tag for full record page (includes alt
    *         attribute text)
    *   $Resource - Resource object
    *
    * And these values may (or may not) be available:
    *   $Description - possibly-truncated text for standard description field
    *         (terms highlighted and inappropriate HTML tags already stripped)
    *   $DisplayableResourceLink - displayable URL for resource itself or file
    *   $ResourceLink - URL to go to resource or to download file
    *   $ResourceLinkTag - complete <a> tag for resource link (includes title for
    *         alt attribute text and target=blank if resource should launch new window)
    *   $Title - text for standard title field (inappropriate HTML tags already stripped)
    */

    $HTMLData = "<div class=\"cw-quicksearch-resource-result\" "
        ."data-resourceid=\"".$Resource->Id()."\">";

    if (isset($Title))
    {
        $HTMLData.=(($G_SysConfig->Value("TitlesLinkTo") == "RECORD")
                    ? $FullRecordLinkTag : (isset($ResourceLinkTag) && $UrlLooksValid
                    ? $ResourceLinkTag : ""));
        $HTMLData.=($Title);
        $HTMLData.=(($G_SysConfig->Value("TitlesLinkTo") == "RECORD")
                    ? "</a>" : (isset($ResourceLinkTag) && $UrlLooksValid
                            ? "</a>" : ""));
    }

    if (isset($Description))
    {
        $HTMLData.='<p>'.$Description.'</p>';
    }

    $HTMLData.='</div>';

    return $HTMLData;
}
